<?php

namespace App\Controller;

use App\Entity\Main\Manager;
use App\Entity\Main\User;
use App\Entity\Main\Visitor;
use App\Form\ManagerType;
use App\Form\VisitorType;
use App\Service\EmailsTemplateSender;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use App\Service\GlobalServices;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\QueryBuilder;
use Omines\DataTablesBundle\Adapter\Doctrine\ORMAdapter;
use Omines\DataTablesBundle\Column\TextColumn;
use Omines\DataTablesBundle\DataTableFactory;
use Symfony\Component\Security\Core\Encoder\EncoderFactoryInterface;


/**
 * @Route("/user")
 */
class UserController extends Controller
{

    protected $em;
    protected $globalServices;
    private $encoderFactory;
    private $emailsTemplateSender;


    /**
     * UserController constructor.
     * @param EntityManagerInterface $entityManager
     * @param GlobalServices $globalServices
     */
    public function __construct(EntityManagerInterface $entityManager,
                                EmailsTemplateSender $emailsTemplateSender,
                                GlobalServices $globalServices,
                                EncoderFactoryInterface $encoderFactory)
    {
        $this->em = $entityManager;
        $this->globalServices = $globalServices;
        $this->emailsTemplateSender = $emailsTemplateSender;
        $this->encoderFactory = $encoderFactory;
    }

    /**
     * Description :: list of managers
     * @Route("/manager_list", name="admin_user_manager_list")
     */
    public function adminUserManagerList(Request $request, DataTableFactory $dataTableFactory)
    {
        if (!$this->getUser()->getManager()) {
            $this->addFlash('danger', "This page is accessible only by platform administrators!");
            return $this->redirectToRoute('error_page');
        }

        if(!$request->query->get('length')){
            $this->get('session')->remove('filter_admin_user_manager_list');
        }

        $datatable = $dataTableFactory->create([])
            ->createAdapter(ORMAdapter::class, [
                'entity' => User::class,
                'query' => function (QueryBuilder $builder) {
                    $builder
                        ->select('manager')
                        ->addSelect('user')
                        ->from(Manager::class, 'manager')
                        ->leftJoin('manager.user', 'user')
                        ->where('user.id != :connected_user')
                        ->setParameter('connected_user', $this->getUser()->getId());
                    if ($this->get('session')->has('filter_admin_user_manager_list')) {
                        $filters = $this->get('session')->get('filter_admin_user_manager_list');
                        if ($filters['email'] != "") $builder->andWhere("LOWER(user.email) LIKE :email")->setParameter('email', '%' . strtolower($filters['email']) . '%');
                        if ($filters['firstName'] != "") $builder->andWhere("LOWER(manager.firstName) LIKE :firstName")->setParameter('firstName', '%' . strtolower($filters['firstName']) . '%');
                        if ($filters['lastName'] != "") $builder->andWhere("LOWER(manager.lastName) LIKE :lastName")->setParameter('lastName', '%' . strtolower($filters['lastName']) . '%');
                    }
                }
            ])
            ->add('lastName', TextColumn::class, [
                'label' => 'Last name',
                'field' => 'manager.lastName'
            ])
            ->add('firstName', TextColumn::class, [
                'label' => 'First name',
                'field' => 'manager.firstName'
            ])
            ->add('email', TextColumn::class, [
                'label' => 'Email',
                'field' => 'user.email'
            ])
            ->add('enabled', TextColumn::class, [
                'label' => 'Status',
                'field' => 'user.enabled',
                'orderable' => true,
                'render' => function ($value, $entity) {
                    return sprintf($entity->getUser()->isEnabled() ? '<span class="badge badge-info"> Enabled </span>' : '<span class="badge badge-warning"> Disabled </span>');
                }
            ])
            ->add('buttons', TextColumn::class, [
                'label' => 'Action',
                'raw' => true,
                'className' => 'dt-action-btns',
                'render' => function ($value, $entity) {
                    $buttons = "";
                    $buttons .= '<a href="' . $this->generateUrl('admin_user_manager_add', ['id' => $entity->getId()]) . '" class="ml-2 text-primary m-r-5" title="Edit"><i class="fas fa-edit"></i></a>';
                    $buttons .= !$entity->getUser()->isEnabled() ? '<a href="javascript:;" class="ml-2 text-primary change-status-action" title="Enable" data-email="' . $entity->getUser()->getEmail() . '" data-status="1"><i class="fa fa-user-check"></i></a>' : '';
                    $buttons .= $entity->getUser()->isEnabled() ? '<a href="javascript:;" class="ml-2 text-primary change-status-action" title="Disable" data-email="' . $entity->getUser()->getEmail() . '" data-status="0"><i class="fa fa-user-minus"></i></a>' : '';
                    return sprintf($buttons);
                }
            ])
            ->handleRequest($request);

        if ($datatable->isCallback()) {
            return $datatable->getResponse();
        }

        return $this->render('user/manager_list.html.twig', [
            'datatable' => $datatable
        ]);
    }

    /**
     * Description :: list of visitors
     * @Route("/visitor_list", name="admin_user_visitor_list")
     */
    public function adminUserVisitorList(Request $request, DataTableFactory $dataTableFactory)
    {
        if (!$this->getUser()->getManager()) {
            $this->addFlash('danger', "This page is accessible only by platform administrators!");
            return $this->redirectToRoute('error_page');
        }

        if(!$request->query->get('length')){
            $this->get('session')->remove('filter_admin_user_visitor_list');
        }

        $datatable = $dataTableFactory->create([])
            ->createAdapter(ORMAdapter::class, [
                'entity' => User::class,
                'query' => function (QueryBuilder $builder) {
                    $builder
                        ->select('visitor')
                        ->addSelect('user')
                        ->from(Visitor::class, 'visitor')
                        ->leftJoin('visitor.user', 'user');
                    if ($this->get('session')->has('filter_admin_user_visitor_list')) {
                        $filters = $this->get('session')->get('filter_admin_user_visitor_list');
                        if ($filters['email'] != "") $builder->andWhere("LOWER(user.email) LIKE :email")->setParameter('email', '%' . strtolower($filters['email']) . '%');
                        if ($filters['firstName'] != "") $builder->andWhere("LOWER(visitor.firstName) LIKE :firstName")->setParameter('firstName', '%' . strtolower($filters['firstName']) . '%');
                        if ($filters['lastName'] != "") $builder->andWhere("LOWER(visitor.lastName) LIKE :lastName")->setParameter('lastName', '%' . strtolower($filters['lastName']) . '%');
                    }
                }
            ])
            ->add('lastName', TextColumn::class, [
                'label' => 'Last name',
                'field' => 'visitor.lastName'
            ])
            ->add('firstName', TextColumn::class, [
                'label' => 'First name',
                'field' => 'visitor.firstName'
            ])
            ->add('email', TextColumn::class, [
                'label' => 'Email',
                'field' => 'user.email'
            ])
            ->add('enabled', TextColumn::class, [
                'label' => 'Status',
                'field' => 'user.enabled',
                'orderable' => true,
                'render' => function ($value, $entity) {

                    return sprintf($entity->getUser()->isEnabled() ? '<span class="badge badge-info"> Enabled </span>' : '<span class="badge badge-warning"> Disabled </span>');
                }
            ])
            ->add('buttons', TextColumn::class, [
                'label' => 'Action',
                'raw' => true,
                'className' => 'dt-action-btns',
                'render' => function ($value, $entity) {
                    $buttons = "";
                    $buttons .= '<a href="' . $this->generateUrl('admin_user_visitor_add', ['id' => $entity->getId()]) . '" class="ml-2 text-primary m-r-5" title="Edit"><i class="fas fa-edit"></i></a>';
                    $buttons .= !$entity->getUser()->isEnabled() ? '<a href="javascript:;" class="ml-2 text-primary change-status-action" title="Enable" data-email="' . $entity->getUser()->getEmail() . '" data-status="1"><i class="fa fa-user-check"></i></a>' : '';
                    $buttons .= $entity->getUser()->isEnabled() ? '<a href="javascript:;" class="ml-2 text-primary change-status-action" title="Disable" data-email="' . $entity->getUser()->getEmail() . '" data-status="0"><i class="fa fa-user-minus"></i></a>' : '';

                    return sprintf($buttons);
                }
            ])
            ->handleRequest($request);

        if ($datatable->isCallback()) {
            return $datatable->getResponse();
        }

        return $this->render('user/visitor_list.html.twig', [
            'datatable' => $datatable
        ]);
    }

    /**
     * Description :: add a new manager
     * @Route("/manager_add/{id}", defaults={"id" = null}, name="admin_user_manager_add")
     */
    public function adminUserManagerAdd(Request $request, $id = null)
    {
        if (!$this->getUser()->getManager()) {
            $this->addFlash('danger', "This page is accessible only by platform administrators!");
            return $this->redirectToRoute('error_page');
        }
        if (!$id) {
            $manager = new Manager();
        }else if (is_object($this->em->getRepository(Manager::class)->find($id))) {
            $manager = $this->em->getRepository(Manager::class)->find($id);
        } else {
            $this->addFlash('danger', "This page does not exist any more!");
            return $this->redirectToRoute('error_page');
        }
        $formManager = $this->createForm(ManagerType::class, $manager)->handleRequest($request);
        if ($request->isXmlHttpRequest()) {
            $registration = $this->globalServices->managerRegistration($manager, $request->request->get("manager"));
            if ($registration["status"] === 200) {
                // Add flash message for the home page after redirection
                $this->addFlash("success", $registration["message"]);
                $this->em->flush();
                if (!$id){
                    $this->emailsTemplateSender->email_registration_temporary_password($registration["user"]);
                }else{
                    $this->emailsTemplateSender->email_update_acount($registration["user"]);
                }
            }

            // Prepare a json response, if status 500 show a notify with a message
            return new JsonResponse([
                'status' => $registration["status"],
                'message' => $registration["message"]
            ]);
        }
        return $this->render('user/manager_add.html.twig', array(
            'manager' => $manager,
            'formManager' => $formManager->createView(),
        ));
    }

    /**
     * Description :: add a new manager
     * @Route("/visitor_add/{id}", defaults={"id" = null}, name="admin_user_visitor_add")
     */
    public function adminUserVisitorAdd(Request $request, $id = null)
    {
        if (!$this->getUser()->getManager()) {
            $this->addFlash('danger', "This page is accessible only by platform administrators!");
            return $this->redirectToRoute('error_page');
        }
        if (!$id) {
            $visitor = new Visitor();
        }else if (is_object($this->em->getRepository(Visitor::class)->find($id))) {
            $visitor = $this->em->getRepository(Visitor::class)->find($id);
        } else {
            $this->addFlash('danger', "This page does not exist any more!");
            return $this->redirectToRoute('error_page');
        }

        $formVisitor = $this->createForm(VisitorType::class, $visitor)->handleRequest($request);
        if ($request->isXmlHttpRequest()) {
            $registration = $this->globalServices->visitorRegistration($visitor, $request->request->get("visitor"));
            if ($registration["status"] === 200) {
                // Add flash message for the home page after redirection
                $this->addFlash("success", $registration["message"]);
                $this->em->flush();

                if (!$id){
                    $this->emailsTemplateSender->email_registration_temporary_password($registration["user"]);
                }else{
                    $this->emailsTemplateSender->email_update_acount($registration["user"]);
                }
            }

            // Prepare a json response, if status 500 show a notify with a message
            return new JsonResponse([
                'status' => $registration["status"],
                'message' => $registration["message"]
            ]);
        }
        return $this->render('user/visitor_add.html.twig', array(
            'visitor' => $visitor,
            'formVisitor' => $formVisitor->createView(),
        ));
    }

    /**
     * Description :: Enable/disable a user
     * @IsGranted({"ROLE_MANAGER"})
     * @Route("/change_status_ajax", name="admin_user_change_status_ajax")
     */
    public function adminUserChangeStatusAjax(Request $request)
    {
        $userManager = $this->container->get('fos_user.user_manager');
        $active = $request->request->get('status') === "1";
        $user = $userManager->findUserByUsernameOrEmail($request->request->get('email'));

        if ($user) {
            $user->setEnabled($active);
            $userManager->updateUser($user);

            return new JsonResponse(array(
                'status' => 200,
                'message' => $active ? "This user is successfully activated." : "This user is successfully deactivated."
            ));
        } else {
            return new JsonResponse(array(
                'status' => 500,
                'message' => "Unknown user!"
            ));
        }
    }

    /**
     * Description :: visitor edit password page
     * @Route("/edit_password", name="edit_password")
     */
    public function visitorEditPassword(Request $request)
    {
        return $this->render('user/edit_password.html.twig');
    }

    /**
     * Description :: visitor edit password page
     * @Route("/edit_password_ajax", name="edit_password_ajax")
     */
    public function visitorEditAjax(Request $request)
    {
        $user = $this->getUser();
        $data = $request->request->get("user");

        if ($data && array_key_exists("oldPassword", $data) && array_key_exists("newPassword", $data)){
            // first check whether the entered password is same as the registered password in the database or not
            // if yes we let the user change his password otherwise we return an error message
            if (!($this->encoderFactory->getEncoder($user)->isPasswordValid($user->getPassword(),$data['oldPassword'], $user->getSalt()))) {
                return new JsonResponse(array(
                    'status' => 500,
                    'message' => "Please check your current password!"
                ));
            } else {
                $user->setPlainPassword($data['newPassword']);
                $user->setTemporaryPassword(null);
                $this->get('fos_user.user_manager')->updateUser($user);
            }
        }
        $this->em->persist($user);
        $this->em->flush();
        return new JsonResponse([
            'status' => 200,
            'message' => "Your password is changed successfully."
        ]);
    }
}