<?php

namespace App\Controller;

use App\DBAL\Demand\DemandDbal;
use App\Entity\Main\Data;
use App\Entity\Main\Demand;
use App\Entity\Main\Site;
use App\Entity\Main\User;
use App\Service\EmailsTemplateSender;
use App\Service\Functions;
use App\Service\GlobalServices;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\QueryBuilder;
use Omines\DataTablesBundle\Adapter\ArrayAdapter;
use Omines\DataTablesBundle\Adapter\Doctrine\ORMAdapter;
use Omines\DataTablesBundle\Column\DateTimeColumn;
use Omines\DataTablesBundle\Column\TextColumn;
use Omines\DataTablesBundle\DataTableFactory;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @Route("/demand")
 */
class DemandController extends AbstractController
{
    protected $em;
    protected $globalServices;
    private $functions;
    private $emailsTemplateSender;

    public function __construct(EntityManagerInterface $entityManager,
                                GlobalServices $globalServices,
                                Functions $functions,
                                EmailsTemplateSender $emailsTemplateSender)
    {
        $this->em = $entityManager;
        $this->globalServices = $globalServices;
        $this->functions = $functions;
        $this->emailsTemplateSender = $emailsTemplateSender;
    }

    /**
     * Description :: list of export requests for user
     * @Route("/list_user", name="user_demand_list")
     */
    public function listUser(Request $request, DataTableFactory $dataTableFactory)
    {
        // check if the current user is a visitor or not
        if (!$this->getUser()->getVisitor()) {
            $this->addFlash('danger', "This page is accessible only by platform users!");
            return $this->redirectToRoute('error_page');
        }
        // check if the user changed his password and validated the cgu
        if ($this->getUser()->getVisitor() && !$this->getUser()->getVisitor()->getValidateCgu()) {
            if ($this->getUser()->getTemporaryPassword()){
                return $this->redirectToRoute('edit_password');
            }else{
                $this->addFlash('danger', "You must accept the terms and conditions and the privacy policy!");
                return $this->redirectToRoute('home');
            }

        }

        $datatable = $dataTableFactory->create([])
            ->createAdapter(ORMAdapter::class, [
                'entity' => Demand::class,
                'query' => function (QueryBuilder $builder) {
                    $builder
                        ->select('d')
                        ->from(Demand::class, 'd')
                        ->leftJoin('d.user', 'u')
                        ->Where('u = :currentUser')
                        ->setParameter('currentUser', $this->getUser()->getId())
                        ->orderBy("d.createdAt", "DESC")
                    ;
                }
            ])
//            ->add('id', TextColumn::class, [
//                'label' => 'Number',
//                'field' => 'd.id'
//            ])

            ->add('createdAt', DateTimeColumn::class, [
                'label' => 'Date',
                'format' => 'd-m-Y H:i',
                'orderable' => false,
            ])

            ->add('buttons', TextColumn::class, [
                'label' => 'Action',
                'raw' => true,
                'className' => 'dt-action-btns',
                'render' => function ($value, $entity) {
                    $buttons = "";
                    $buttons .= '<a href="' . $this->generateUrl('demand_detail', ['id' =>  $entity->getId()]) .'"  class="ml-2 text-primary m-r-5 "  title="Show"><i class="fas fa-eye"></i></a>';
                    return sprintf($buttons);
                }
            ])
            ->handleRequest($request);

        if ($datatable->isCallback()) {
            return $datatable->getResponse();
        }

        return $this->render('demand/list_user.html.twig', [
            'datatable' => $datatable
        ]);
    }

    /**
     * Description :: list of all export requests
     * @Route("/list_admin", name="admin_demand_list")
     */
    public function listAdmin (Request $request, DataTableFactory $dataTableFactory)
    {
        // check if the current user is a manager or not
        if (!$this->getUser()->getManager()) {
            $this->addFlash('danger', "This page is accessible only by platform administrators!");
            return $this->redirectToRoute('error_page');
        }

        $datatable = $dataTableFactory->create([])
            ->createAdapter(ORMAdapter::class, [
                'entity' => Demand::class,
                'query' => function (QueryBuilder $builder) {
                    $builder
                        ->select('d')
                        ->from(Demand::class, 'd')
                        ->leftJoin('d.user', 'u')
                        ->orderBy("d.createdAt", "DESC")
                        ->addOrderBy("d.checked", "DESC")
                    ;
                }
            ])
            ->add('id', TextColumn::class, [
                'label' => 'User',
                'field' => 'd.user',
                'orderable' => false,
            ])

            ->add('createdAt', DateTimeColumn::class, [
                'label' => 'Date',
                'format' => 'd-m-Y H:i',
                'orderable' => false,
            ])

            ->add('buttons', TextColumn::class, [
                'label' => 'Action',
                'raw' => true,
                'className' => 'dt-action-btns',
                'render' => function ($value, $entity) {
//                dd($this->getParameter('absolute_upload_export_data_directory')."/". $entity->getFileExport());
                    $buttons = "";
                    $buttons .= '<a href="' . $this->generateUrl('demand_detail', ['id' =>  $entity->getId()]) .'"  class="ml-2 text-primary m-r-5 "  title="Show"><i class="fas fa-eye"></i></a>';
                    if($entity->getFileExport() && file_exists($this->getParameter('absolute_upload_export_data_directory')."/". $entity->getFileExport()) == true){
                        //$buttons .= '<a href= "'. $this->getParameter('relative_upload_export_data_directory')."/". $entity->getFileExport() .'" class="ml-2 text-primary " title="Download" target="_blank"><i class="fas fa-file-download"></i></a>';
                    }
                    return sprintf($buttons);
                }
            ])

//            ->add('detail', TextColumn::class, [
//                'label' => 'Detail',
//                'raw' => true,
//                'className' => 'dt-action-btns',
//                'render' => function ($value, $entity) {
//                    $buttons = "";
//                    $buttons .= '<a target="_bank" href="' . $this->generateUrl('data_list') .'?filter[age]='.$entity->getAge().'&filter[season]='.$entity->getSeason().'&filter[zone]='.$entity->getZone().'"
//                                class="ml-2 text-primary m-r-5 "  title="Show"><i class="fa-solid fa-angles-right"></i></a>';
//
//                    return sprintf($buttons);
//                }
//            ])

            ->add('export', TextColumn::class, [
                'label' => 'Export',
                'raw' => true,
                'className' => 'dt-action-btns',
                'render' => function ($value, $entity) {
                    $buttons = "";

                    // dans le cas ou le fichier a été générer et disponible sur le serveur on donne la main pour téléchargement dans le cas contraire on affiche une icône désactivée
                    if($entity->getFileExport() != '' && $entity->getFileExport() != null){
                        $url = $this->getParameter('URL_FILE').$entity->getFileExport();
                        $buttons .= ' <a href="' .$url.'" download="'.$entity->getFileExport().'" 
                                class="ml-2 text-primary m-r-5 "  title="Show"><i class="fa-solid fa-file-arrow-down"></i></a>';
                    }else{
                        $buttons .= ' <a href="javascript:void(0)"  class="ml-2 text-primary m-r-5 disabled"  title="Show"><i class="fa-solid fa-file-arrow-down"></i></a>';
                    }

                    return sprintf($buttons);
                }
            ])
            ->handleRequest($request);

        if ($datatable->isCallback()) {
            return $datatable->getResponse();
        }

        return $this->render('demand/list_admin.html.twig', [
            'datatable' => $datatable
        ]);
    }

    /**
     * Description :: send data export request
     * @Route("/request_data_ajax", name="request_data_ajax")
     */
    public function requestDataAjax(Request $request)
    {
        $recap = $request->get('recap');

        // get all details of the new demand of request data of the user
        $zones = $this->em->getRepository(Site::class)->findBy(array('id' => explode(',', $recap["zone"])));
        $zoneArray = array();
        foreach ($zones as $zone){
            array_push($zoneArray,$zone->getZone());
        }

        $checkedArray = array();
        $seasonArray = array();
        $listSeasonsAsString = "";

        foreach (json_decode($recap["checked"], true) as $checked){
            $zone = $this->em->getRepository(Site::class)->find(array('id' => $checked[1]));
            array_push($checkedArray, $checked[0] . ' / ' . str_replace($checked[1],$zone->getZone(),$checked[1]));
            array_push($seasonArray, $checked[0]);
        }

        $user = $this->getUser()->getVisitor();
        if(!empty($seasonArray)){
            $result = implode(",", $seasonArray);
            $uniqueSeasons = array_unique(explode(',', $result));
            $listSeasonsAsString = implode(",", $uniqueSeasons);
        }

        //sending a notification email to all the managers of the application to notify of the new user request
        $this->emailsTemplateSender->email_request_data($user,
            $this->em->getRepository(User::class)->getUsersByRole("ROLE_MANAGER"),
            implode(' / ',$zoneArray),
            $recap["season"],
            $recap["age"],
            $checkedArray);

        // create a new demand of request data
        $demand = new Demand();
        if ($listSeasonsAsString != ""){
            $demand->setSeason($listSeasonsAsString);
        }
        $recap["zone"] ? $demand->setZone($recap["zone"]) : null;
        $recap["age"] ? $demand->setAge($recap["age"]): null;
        $demand->setState(DemandDbal::WAITING);
        $demand->setUser($this->getUser());
        $demand->setChecked($recap["checked"]);
        $this->em->persist($demand);
        $this->em->flush();

        // Génération de fichiers + enregistrement dans la base de données (preparation du fichier Csv) sans l'envoi de mail vers l'utilisateur ni l'admin
        // once the file is generated, the administrator sends it manually by email to the user
        $path = $this->getParameter("PATH_EXPORT");

        //Dev
//        exec('/usr/bin/nohup php7.3 '. $path .'/bin/console app:export '. $demand->getId() . ' no '.str_replace(" ", "_", $user->getFullName())  .' >/dev/null 2>&1 &');

        // Prod
        exec('/usr/bin/nohup php ' . $path . '/bin/console app:export ' . $demand->getId() . ' no ' . str_replace(" ", "_", $user->getFullName()) . ' >/dev/null 2>&1 &');

        return new JsonResponse([
            'status' => 200,
            'message' => 'An e-mail has been sent to the administrators containing the details of your request.'
        ]);
    }

    /**
     * Description ::  refuse the request
     * @Route("/demand_refuse_ajax", name="demand_refuse_ajax")
     */
    public function demandRefuse(Request $request)
    {
       $demand = $this->em->getRepository(Demand::class)->find($request->request->get("id"));

        $demand->setState(DemandDbal::REFUSED);

        $this->em->persist($demand);
        $this->em->flush();

       return new JsonResponse(array(
            'status' => 200,
            'message' => "Export request has been successfully refused"
        ));
    }

    /**
     * Description ::   the request detail
     * @Route("/demand_detail/{id}", name="demand_detail")
     */
    public function demandDetail(Request $request, $id)
    {
        $demand= $this->em->getRepository(Demand::class)->find($id);

        // check if the demand already exists or not
        if (!is_object($demand)) {
            $this->addFlash('danger', "This page does not exist any more!");
            return $this->redirectToRoute('error_page');
        }

        // get the details of the selected filter (zone, age, seasons, and which season checked)in the request.
        $zones = $this->em->getRepository(Site::class)->findBy(array('id' => explode(',', $demand->getZone())));
        $zoneArray = array();
        foreach ($zones as $zone){
            array_push($zoneArray,$zone->getZone());
        }

        $all_checked = json_decode($demand->getChecked());
        $checkedArray = array();
        foreach($all_checked as $checked){
            $zone = $this->em->getRepository(Site::class)->find(array('id' => $checked[1]));
            array_push($checkedArray, array("season"=>$checked[0], "zone"=>$zone->getZone()));
        }
        return $this->render('demand/detail.html.twig',[
            'zones' => implode(' / ',$zoneArray),
            'seasons' => $demand->getSeason(),
            'age' => $demand->getAge(),
            'all_checked' => $checkedArray
        ]);
    }
}
